<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\VirtualCardCategory;
use App\Models\VirtualCardApplication;
use App\Models\VirtualCard;
use App\Models\Settings;
use Illuminate\Http\Request;

class VirtualCardCategoryController extends Controller
{
    public function index()
    {
        $categories = VirtualCardCategory::paginate(20);
        return view('admin.virtual-card-categories.index', compact('categories'))->with([
            'title' => 'Card Categories',
            'settings' => Settings::first()
        ]);
    }

    public function create()
    {
        return view('admin.virtual-card-categories.create')->with([
            'title' => 'Create Card Category',
            'settings' => Settings::first()
        ]);
    }

    public function store(Request $request)
    {
        $request->validate([
            'name' => 'required|string|max:255',
            'description' => 'required|string',
            'price' => 'required|numeric|min:0',
            'daily_limit' => 'required|numeric|min:0',
            'monthly_limit' => 'required|numeric|min:0',
            'annual_fee' => 'required|numeric|min:0',
        ]);

        VirtualCardCategory::create($request->all());
        return redirect()->route('admin.virtual-card-categories.index')->with('success', 'Category created successfully!');
    }

    public function applications()
    {
        $applications = VirtualCardApplication::with(['user', 'category'])->latest()->paginate(20);
        return view('admin.virtual-card-applications.index', compact('applications'))->with([
            'title' => 'Card Applications',
            'settings' => Settings::first()
        ]);
    }

    public function approveApplication($id)
    {
        $application = VirtualCardApplication::findOrFail($id);
        
        // Create virtual card
        $card = VirtualCard::create([
            'user_id' => $application->user_id,
            'card_number' => $this->generateCardNumber($application->preferred_card_type),
            'card_holder_name' => strtoupper($application->full_name),
            'expiry_month' => '12',
            'expiry_year' => date('Y') + 3,
            'cvv' => rand(100, 999),
            'balance' => 0.00,
            'daily_limit' => $application->category->daily_limit,
            'monthly_limit' => $application->category->monthly_limit,
            'card_type' => $application->preferred_card_type
        ]);

        // Update application
        $application->update([
            'status' => 'approved',
            'virtual_card_id' => $card->id
        ]);

        return redirect()->back()->with('success', 'Application approved and card generated!');
    }

    public function rejectApplication(Request $request, $id)
    {
        $application = VirtualCardApplication::findOrFail($id);
        
        // Refund the application fee
        $application->user->increment('account_bal', $application->category->price);
        
        $application->update([
            'status' => 'rejected',
            'admin_notes' => $request->notes
        ]);

        return redirect()->back()->with('success', 'Application rejected and fee refunded!');
    }

    public function viewApplication($id)
    {
        $application = VirtualCardApplication::with(['user', 'category'])->findOrFail($id);
        return view('admin.virtual-card-applications.show', compact('application'))->with([
            'title' => 'Application Details',
            'settings' => Settings::first()
        ]);
    }

    public function toggleStatus($id)
    {
        $category = VirtualCardCategory::findOrFail($id);
        $category->update([
            'status' => $category->status == 'active' ? 'inactive' : 'active'
        ]);
        return redirect()->back()->with('success', 'Category status updated successfully!');
    }

    private function generateCardNumber($cardType = 'visa')
    {
        switch ($cardType) {
            case 'visa':
                return '4' . str_pad(rand(0, 999999999999999), 15, '0', STR_PAD_LEFT);
            case 'mastercard':
                return '5' . str_pad(rand(0, 999999999999999), 15, '0', STR_PAD_LEFT);
            case 'american_express':
                return '3' . str_pad(rand(0, 99999999999999), 14, '0', STR_PAD_LEFT);
            default:
                return '4' . str_pad(rand(0, 999999999999999), 15, '0', STR_PAD_LEFT);
        }
    }
}