<?php

namespace App\Http\Controllers\User;

use App\Http\Controllers\Controller;
use App\Models\VirtualCard;
use App\Models\VirtualCardTransaction;
use App\Models\VirtualCardCategory;
use App\Models\VirtualCardApplication;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;

class VirtualCardController extends Controller
{
    public function index()
    {
        $cards = VirtualCard::where('user_id', Auth::id())->get();
        $applications = VirtualCardApplication::where('user_id', Auth::id())->with('category')->latest()->get();
        return view('user.virtual-cards.index', compact('cards', 'applications'))->with([
            'title' => 'Virtual Cards'
        ]);
    }

    public function categories()
    {
        $categories = VirtualCardCategory::where('status', 'active')->get();
        return view('user.virtual-cards.categories', compact('categories'))->with([
            'title' => 'Card Categories'
        ]);
    }

    public function apply($categoryId)
    {
        $category = VirtualCardCategory::findOrFail($categoryId);
        return view('user.virtual-cards.apply', compact('category'))->with([
            'title' => 'Apply for Card'
        ]);
    }

    public function submitApplication(Request $request)
    {
        $request->validate([
            'category_id' => 'required|exists:virtual_card_categories,id',
            'full_name' => 'required|string|max:255',
            'phone' => 'required|string|max:20',
            'address' => 'required|string',
            'city' => 'required|string|max:100',
            'state' => 'required|string|max:100',
            'zip_code' => 'required|string|max:20',
            'preferred_card_type' => 'required|in:visa,mastercard,american_express',
            'purpose' => 'required|string'
        ]);

        $category = VirtualCardCategory::findOrFail($request->category_id);
        $user = Auth::user();

        if ($user->account_bal < $category->price) {
            return redirect()->back()->with('error', 'Insufficient balance! You need $' . number_format($category->price, 2) . ' to apply for this card.');
        }

        DB::transaction(function () use ($request, $category, $user) {
            // Debit application fee
            $user->decrement('account_bal', $category->price);

            // Create application
            VirtualCardApplication::create([
                'user_id' => $user->id,
                'virtual_card_category_id' => $category->id,
                'full_name' => $request->full_name,
                'phone' => $request->phone,
                'address' => $request->address,
                'city' => $request->city,
                'state' => $request->state,
                'zip_code' => $request->zip_code,
                'preferred_card_type' => $request->preferred_card_type,
                'purpose' => $request->purpose,
                'status' => 'pending'
            ]);
        });

        return redirect()->route('user.virtual-cards.index')->with('success', 'Application submitted successfully! Fee of $' . number_format($category->price, 2) . ' has been deducted.');
    }

    public function fundCard(Request $request, $id)
    {
        $request->validate(['amount' => 'required|numeric|min:1']);
        
        $card = VirtualCard::where('id', $id)->where('user_id', Auth::id())->firstOrFail();
        $user = Auth::user();
        
        if ($user->account_bal < $request->amount) {
            return redirect()->back()->with('error', 'Insufficient balance!');
        }

        DB::transaction(function () use ($card, $user, $request) {
            $user->decrement('account_bal', $request->amount);
            $card->increment('balance', $request->amount);
            
            VirtualCardTransaction::create([
                'virtual_card_id' => $card->id,
                'type' => 'credit',
                'amount' => $request->amount,
                'description' => 'Card funding from account balance',
                'reference' => 'FUND_' . time(),
                'status' => 'completed'
            ]);
        });

        return redirect()->back()->with('success', 'Card funded successfully!');
    }

    public function withdrawCard(Request $request, $id)
    {
        $request->validate(['amount' => 'required|numeric|min:1']);
        
        $card = VirtualCard::where('id', $id)->where('user_id', Auth::id())->firstOrFail();
        $user = Auth::user();
        
        if ($card->balance < $request->amount) {
            return redirect()->back()->with('error', 'Insufficient card balance!');
        }

        DB::transaction(function () use ($card, $user, $request) {
            $card->decrement('balance', $request->amount);
            $user->increment('account_bal', $request->amount);
            
            VirtualCardTransaction::create([
                'virtual_card_id' => $card->id,
                'type' => 'debit',
                'amount' => $request->amount,
                'description' => 'Card withdrawal to account balance',
                'reference' => 'WITHDRAW_' . time(),
                'status' => 'completed'
            ]);
        });

        return redirect()->back()->with('success', 'Amount withdrawn successfully!');
    }

    public function block($id)
    {
        VirtualCard::where('id', $id)->where('user_id', Auth::id())
            ->update(['status' => 'blocked']);
        
        return redirect()->back()->with('success', 'Card blocked successfully!');
    }

    public function unblock($id)
    {
        VirtualCard::where('id', $id)->where('user_id', Auth::id())
            ->update(['status' => 'active']);
        
        return redirect()->back()->with('success', 'Card unblocked successfully!');
    }

    private function generateCardNumber()
    {
        return '4' . str_pad(rand(0, 999999999999999), 15, '0', STR_PAD_LEFT);
    }
}